#region Includes
using GroundhogLearning.GameKit.Utils;
using UnityEngine;
using UnityEngine.InputSystem;
#endregion

namespace GroundhogLearning.GameKit.Interactions
{
    [AddComponentMenu(EditorMenuConfig.CATEGORY_INTERACTIONS + "Ray Interactor")]
    public class RayInteractor : MonoBehaviour
    {
        #region Variables

        [Header("References")]
        [SerializeField] private Transform _attachTransform;

        [Header("Configuration")]

        [Tooltip("The maximum distance for ray detection.")]
        [SerializeField] private float _maxDetectionDistance;

        [Tooltip("The minimum distance for ray detection.")]
        [SerializeField] private float _minDetectionDistance;

        [Tooltip("Layer mask for raycasting.")]
        [SerializeField] private LayerMask _layerMask;

        [Tooltip("Primary action used to activate an interactable (e.g. Interact).")]
        [SerializeField] private InputActionReference _primaryAction;

        [Tooltip("Secondary action used to activate an interactable (e.g. AltInteract).")]
        [SerializeField] private InputActionReference _secondaryAction;

        private Ray _ray;
        private RaycastHit _hit;

        private Interactable _detectedInteractable;

        private Interactable _attachedInteractable;

        #endregion

        private void Update()
        {
            var interactable = GetInteractable();
            var hasInput = GetInput();

            if (interactable == null)
            {
                ResetInteraction();

                if (hasInput && _attachedInteractable != null)
                {
                    DetachInteractable();
                }
            }
            else
            {
                if (_detectedInteractable != null && interactable != _detectedInteractable)
                {
                    ResetInteraction();
                }

                if (_detectedInteractable == null)
                {
                    _detectedInteractable = interactable;
                    _detectedInteractable.Enter(this);
                }
                else
                {
                    _detectedInteractable.Stay(this);
                }

                if (hasInput)
                {
                    var interactableToAttach = _detectedInteractable.Activate(this);
                    if (interactableToAttach != null)
                    {
                        AttachInteractable(interactable);
                    }
                    else if (_attachedInteractable != null)
                    {
                        DetachInteractable();
                    }
                }
            }
        }

        private Interactable GetInteractable()
        {
            _ray = new Ray(transform.position, transform.forward);
            if (Physics.Raycast(_ray, out _hit, _maxDetectionDistance, _layerMask))
            {
                var distance = Vector3.Distance(transform.position, _hit.transform.position);
                if (distance > _minDetectionDistance)
                {
                    if (_hit.collider.transform.TryGetComponent<Interactable>(out var interactable))
                    {
                        return interactable;
                    }
                }
            }
            return null;
        }

        private void ResetInteraction()
        {
            if (_detectedInteractable != null)
            {
                _detectedInteractable.Exit(this);
                _detectedInteractable = null;
            }
        }

        private void AttachInteractable(Interactable interactable)
        {
            if (interactable == null) return;
            _attachedInteractable = interactable;
            _attachedInteractable.transform.SetParent(_attachTransform);
            _attachedInteractable.transform.localPosition = Vector3.zero;
        }
        private void DetachInteractable()
        {
            if (_attachedInteractable == null) return;

            _attachedInteractable.transform.SetParent(null);
            _attachedInteractable.Activate(this);
            _attachedInteractable = null;
        }

        private bool GetInput()
        {
            return (_primaryAction != null && _primaryAction.action.WasPressedThisFrame()) ||
                   (_secondaryAction != null && _secondaryAction.action.WasPressedThisFrame());
        }

#if UNITY_EDITOR
        private void OnDrawGizmosSelected()
        {
            Gizmos.color = Color.yellow;
            Gizmos.DrawRay(transform.position, transform.forward * _maxDetectionDistance);
        }
#endif
    }
}